#ifdef WIN32

#include "JabberWerxCommonHelper.h"

#include "Dependence.h"
#include "jcfcoreutils/StringUtils.h"

#include "csfunified/services/interface/SystemService.h"
#include "csfunified/services/interface/ContactService.h"
#include "csfunified/services/interface/ConfigService.h"
#include "csfunified/services/interface/ConfigValue.h"
#include "csfunified/services/interface/Presence.h"
#include "csfunified/services/interface/PresenceService.h"
#include "csfunified/services/interface/PresenceServiceCapabilities.h"

#include "csfunified/services/interface/PresenceOption.h"
#include "csfunified/services/interface/Credentials.h"
#include "csfsystemmonitor/SystemMonitor.h"
#include "csfsystemmonitor/SystemMonitorFactory.h"
#include "csf/logger/CSFLogger.hpp"

#include "services/impl/JabberServiceProvider.h"
#include "services/impl/JabberServiceSink.h"
#include "services/impl/desktopshare/DesktopSharingControllerImplWin.h"

#include "jcfcoreutils/TimeUtils.h"

#include "libxml/parser.h"
#include "libxml/tree.h"

using namespace CSFUnified;

static CSFLogger* JabberWerxCommonHelperLogger = CSFLogger_getLogger("MeetingService-JabberWerxCommonHelperLogger");

const static std::wstring WebEx_DS_Token = L"WebEx_DS_Token";
const static std::wstring WebEx_DS_Url = L"WebEx_DS_Url";
const static std::wstring CUP = L"CUP";
const static std::wstring WBX = L"WebEx";

namespace CSFUnified
{
	JabberWerxCommonHelper::JabberWerxCommonHelper(void)
	{
		m_TimerHelper = NULL;
		SetEnableSpecialInWebExMeetingPresence(false);

		m_pJabberServiceProvider = &JabberServiceProvider::getInstance();
        m_pJabberServiceSink = &JabberServiceSink::getInstance();
		m_pMeetingServiceObjectManager = &MeetingServiceObjectManager::getInstance();
	}

	JabberWerxCommonHelper& JabberWerxCommonHelper::getInstance()
	{
		static JabberWerxCommonHelper s_commonHelper;
		return s_commonHelper;
	}

	JabberWerxCommonHelper::~JabberWerxCommonHelper(void)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::JabberWerxCommonHelper leave.");

		if (m_TimerHelper != NULL)
		{
			csf::TimerHelper::invalidateTimer(m_TimerHelper);
			m_TimerHelper = NULL;
		}

		//CSFSystemMonitor::SystemMonitorFactory::getInstance()->removeNetworkObserver(m_pJabberServiceSink);
	}

	wstring JabberWerxCommonHelper::GetMyAccountName()
	{
		return JCFCoreUtils::toWideString(GetUserName());
	}

	string JabberWerxCommonHelper::GetUserName()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetUserName.");
		std::string JID = m_pJabberServiceProvider->getMyAccount();
	
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetUserName." << JID);

		return JID;
	}

	wstring JabberWerxCommonHelper::GetMyAccountDisplayName()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetMyAccountDisplayName.");
		std::string strTemp = m_pJabberServiceProvider->getMyDisplayName();
		
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetMyAccountDisplayName." << strTemp);

		return JCFCoreUtils::toWideString(strTemp);
	}

	wstring JabberWerxCommonHelper::GetMyAccountEmail()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetMyAccountEmail.");
		std::string strTemp = m_pJabberServiceProvider->getMyEmail();

		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetMyAccountEmail." << strTemp);

		return JCFCoreUtils::toWideString(strTemp);
	}

	wstring JabberWerxCommonHelper::GetBuddyDispName(const wstring& strBuddyName)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetBuddyDispName.");
	
		string str = JCFCoreUtils::toString(strBuddyName);


		std::string sName = m_pJabberServiceProvider->getDisplayNameByJabberId(JCFCoreUtils::toString(strBuddyName));
		std::wstring wName = JCFCoreUtils::toWideString(sName);
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetBuddyDispName." << JCFCoreUtils::toString(wName));
		return wName;
	}

	wstring JabberWerxCommonHelper::GetBuddyEmail(const wstring &strBuddyName)
	{
		string strName = JCFCoreUtils::toString(JCFCoreUtils::unescapeJid(strBuddyName));

		std::string strEmail = m_pJabberServiceProvider->getEmailByJabberId(strName);
		std::wstring wstrEmail = JCFCoreUtils::toWideString(strEmail);

		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetBuddyEmail.");

		return wstrEmail;
	}

	wstring JabberWerxCommonHelper::GetBuddyURIWithEmailaddress(const wstring &strEmailaddress)
	{
		std::string strUri = m_pJabberServiceProvider->getJIDByEmail(JCFCoreUtils::toString(strEmailaddress));
		std::wstring wstrURI = JCFCoreUtils::toWideString(strUri);
	
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetBuddyURIWithEmailaddress.");

		return wstrURI;
	}

	BOOL JabberWerxCommonHelper::IsAccount(const wstring& strSessionName)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsAccount.");
		BOOL bIsAccount = ATFalse;
		if (m_UnifiedFactory.get() != NULL)
		{		
			std::string JID = GetUserName();
			if (JID == JCFCoreUtils::toString(strSessionName))
			{
				bIsAccount = ATTrue;
			}
		}
		return bIsAccount;
	}

	BOOL JabberWerxCommonHelper::IsSignedOn()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsSignedOn.");
		BOOL bSignOn = ATTrue;
		/*if (NULL != m_pClientMgr)
		{
		m_pClientMgr->IsSignedOn(&bSignOn);
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsSignedOn." << bSignOn);
		}*/
		return bSignOn;
	}

	BOOL JabberWerxCommonHelper::IsDSPolicyEnable(const wstring& strSessionName, int& nPolicyResult)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsDSPolicyEnable.");
		bool vBool = m_pJabberServiceProvider->isDSPolicyEnable(JCFCoreUtils::toString(strSessionName), nPolicyResult);
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsDSPolicyEnable(). strSessionName: " << JCFCoreUtils::toString(strSessionName).c_str()
			<< " nPolicyResult: " << nPolicyResult
			<< " vBool: " << (vBool ? "true":"false"));

		return (BOOL)vBool;
	}

	BOOL JabberWerxCommonHelper::AutoRespPolicy(wstring const& strBuddyName, int nPolicyResult)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AutoRespPolicy.");
		CDependence::getInstance()->GetIMMgr().SendAutoRespPolicy(strBuddyName, tagClientPolicyResult(nPolicyResult));
		return ATTrue;
	}

	BOOL JabberWerxCommonHelper::HasDSCapability(const wstring& strSessionName)
	{
		BOOL bEnable = m_pJabberServiceProvider->hasDSCapability(JCFCoreUtils::toString(strSessionName));
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::HasDSCapability." << " bEnable: " << (bEnable ? "true" : "false"));
		return bEnable;
	}

	BOOL JabberWerxCommonHelper::HasIMCapability(const wstring& strSessionName)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::HasIMCapability.");
		
		BOOL bEnable = m_pJabberServiceProvider->hasIMCapability(JCFCoreUtils::toString(strSessionName));

		return bEnable;
	}

	BOOL JabberWerxCommonHelper::IsBuddyOnline(const wstring& strSessionName)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsBuddyOnline.");
		
		BOOL bOnline = m_pJabberServiceProvider->isBuddyOnline(JCFCoreUtils::toString(strSessionName));

		return bOnline;
	}
	
	BOOL JabberWerxCommonHelper::IsWbxMeetingEnabled()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsWbxMeetingEnabled.");
		
		BOOL bEnabled = m_pJabberServiceProvider->isWbxMeetingEnabled();
		
		if (bEnabled) // Check that meetings are enabled
		{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsWbxMeetingEnabled. ATTrue");
			return ATTrue;
		}else{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsWbxMeetingEnabled. ATFalse");
			return ATFalse;
		}
	}

	wstring JabberWerxCommonHelper::GetDSURL()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetDSURL.");
		std::string strTemp = m_pJabberServiceProvider->getDSURL();
		
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetDSURL. strTemp: " << strTemp.c_str());

		return JCFCoreUtils::toWideString(strTemp);
	}


	wstring JabberWerxCommonHelper::GetDSToken()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetDSToken.");
		
		std::string dsToken = m_pMeetingServiceObjectManager->getDesktopSharingController()->GetDSToken();

		return JCFCoreUtils::toWideString(dsToken);
	}

	wstring JabberWerxCommonHelper::GetDSParam()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetDSParam.");
		
		return GetDSToken();
	}


	BOOL JabberWerxCommonHelper::ProbeP2PDnd(wstring const& strBuddyName)
	{
		return ATFalse;
	}

	BOOL JabberWerxCommonHelper::AutoRespDnd(wstring const& strBuddyName)
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AutoRespDnd.");
		return ATTrue;
	}

	void JabberWerxCommonHelper::AddInMeetingStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddInMeetingStatus.");
		m_pJabberServiceProvider->addInMeetingStatus();
 	}

	void JabberWerxCommonHelper::RemoveInMeetingStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::RemoveInMeetingStatus.");
		m_pJabberServiceProvider->removeInMeetingStatus();
	}

	void JabberWerxCommonHelper::AddInWebExMeetingStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddInWebExMeetingStatus.");

		m_pJabberServiceProvider->addInWebExMeetingStatus();
	}

	void JabberWerxCommonHelper::RemoveInWebExMeetingStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::RemoveInWebExMeetingStatus.");

		m_pJabberServiceProvider->removeInWebExMeetingStatus();
	}

	void JabberWerxCommonHelper::DispatchChangeInWebExMeetingStatus(bool isInWebExMeeting)
	{
		if (NULL == m_UnifiedFactory)
		{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::DispatchChangeInWebExMeetingStatus.  m_UnifiedFactory == NULL");
			return;
		}

		bool isOnDispatcher = m_UnifiedFactory->getServicesDispatcher()->checkForUpdateAccess();

		CSFLogDebugS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::DispatchChangeInWebExMeetingStatus");

		if(isOnDispatcher)
		{
			ChangeInWebExMeetingStatus(isInWebExMeeting);
		}
		else
		{
			csf::TaskPtr task(new JCFCoreUtils::FunctionTask(
			   std::bind(&JabberWerxCommonHelper::ChangeInWebExMeetingStatus, this, isInWebExMeeting),
			   "JabberWerxCommonHelper::ChangeInWebExMeetingStatus"));

			m_UnifiedFactory->getServicesDispatcher()->enqueueBlock(task);
		}
	}

	void JabberWerxCommonHelper::ChangeInWebExMeetingStatus(bool isInWebExMeeting)
	{
		if (NULL == m_UnifiedFactory)
		{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::ChangeInWebExMeetingStatus.  m_UnifiedFactory == NULL");
			return;
		}

		SMART_PTR_NS::shared_ptr<PresenceService> services = m_UnifiedFactory->getService<PresenceService>();
		if (NULL == services)
		{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::ChangeInWebExMeetingStatus.  services == NULL");
			return;
		}

		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::ChangeInWebExMeetingStatus " << isInWebExMeeting);
		services->UpdateRichPresenceStatus(RichPresenceStatusEnum::InWebexMeeting, isInWebExMeeting);
	}

	void JabberWerxCommonHelper::SetEnableSpecialInWebExMeetingPresence(bool isEnable)
	{
		const int MEETING_CHECK_STATUS_TIMER = 1000;

		m_isRealInWebExMeeting = false;
		m_isInWebExMeeting = false;
		m_isToday = false;
		m_isSpecialInWebExMeetingPresenceEnable = isEnable;

		{
			csf::ScopedWriteRWLock lock(mLock);
			m_WebExMeetingTime.clear();
		}

		if (m_TimerHelper != NULL)
		{
			csf::TimerHelper::invalidateTimer(m_TimerHelper);
			m_TimerHelper = NULL;
		}

		if (m_isSpecialInWebExMeetingPresenceEnable)
		{
			csf::TimerHelper::scheduleTimer(m_TimerHelper, this, MEETING_CHECK_STATUS_TIMER, true);
		}
	}

	void JabberWerxCommonHelper::onTimerFired(csf::TimerHelper *timer)
	{
		if (!m_isSpecialInWebExMeetingPresenceEnable)
		{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::onTimerFired Timer should be stopped after special logic diabled");
			return;
		}

		if (isInOnlineMeeting() || m_isRealInWebExMeeting)
		{
			if (!m_isInWebExMeeting)
			{
				DispatchChangeInWebExMeetingStatus(true);
				m_isInWebExMeeting = true;
			}
		}
		else
		{
			if (m_isInWebExMeeting)
			{
				DispatchChangeInWebExMeetingStatus(false);
				m_isInWebExMeeting = false;
			}
		}
	}

	void JabberWerxCommonHelper::SetListMeetingDate(const SYSTEMTIME* const sysTime)
	{
		m_isToday = false;
		if (sysTime == NULL)
		{
			m_isToday = true;
		}
		else
		{
#ifdef _WIN32
			SYSTEMTIME stNow;
			GetLocalTime(&stNow);
			if (stNow.wYear == sysTime->wYear && stNow.wMonth == sysTime->wMonth && stNow.wDay == sysTime->wDay)
			{
				m_isToday = true;
			}
#endif
		}
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::SetListMeetingData to " << m_isToday);
	}

	void JabberWerxCommonHelper::onUpdateMeetingList(const wstring& meetingListData)
	{
		if (!m_isToday)
		{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::onUpdateMeetingList Not today meeting list");
			return;
		}

		csf::ScopedWriteRWLock lock(mLock);

		m_WebExMeetingTime.clear();

		ParseMeetingList(meetingListData);
	}

	void JabberWerxCommonHelper::ParseMeetingList(const wstring& meetingListData)
	{
		// construct meeting dom
		std::string strBuffer = JCFCoreUtils::toString(meetingListData);
		xmlDoc* pMeetingDom = xmlReadMemory(strBuffer.c_str(), strBuffer.length(), NULL, NULL, 0);
		if (pMeetingDom == NULL)
		{
			return;
		}

		// get root (WebEx_Connect) element
		xmlNode* pRoot = xmlDocGetRootElement(pMeetingDom);
		if (pRoot == NULL)
		{
			return;
		}

		// get top (Meetings) element
		xmlNode* pTop = pRoot->children;
		if (pTop == NULL || !xmlStrEqual(pTop->name, (const xmlChar*)"Meetings"))
		{
			return;
		}

		for (xmlNode* pCurrent=pTop->children; pCurrent!=NULL; pCurrent=pCurrent->next)
		{
			if (pCurrent->type != XML_ELEMENT_NODE || pCurrent->children == NULL)
			{
				continue;
			}

			if (xmlStrEqual(pCurrent->name, (const xmlChar*)"Meeting"))
			{
				// parse meeting info
				long nStartTime=0, nEndTime=0;
				if (ParseWBXMeetingStartEndTime(pCurrent, nStartTime, nEndTime))
				{
					m_WebExMeetingTime.push_back(std::make_pair(nStartTime, nEndTime));
				}
			}
		}

		// free meeting dom
		xmlFreeDoc(pMeetingDom);
	}

	bool JabberWerxCommonHelper::ParseWBXMeetingStartEndTime(void* pDetail, long& lStartTime, long& lEndTime)
	{
		if (pDetail == NULL)   return false;

		lStartTime = 0; lEndTime = 0;
		long lMeetingType = -1;

		xmlNode* pMeeting = (xmlNode*)pDetail;
		for (xmlNode* pCurrent=pMeeting->children; pCurrent!=NULL; pCurrent=pCurrent->next)
		{
			if (pCurrent->type != XML_ELEMENT_NODE || pCurrent->children == NULL)
			{
				continue;
			}

			// meeting type
			if (xmlStrEqual(pCurrent->name, (const xmlChar*)"EventType"))
			{
				const char* strMeetingType = (const char*)pCurrent->children->content;
				if (strMeetingType != NULL)
				{
					lMeetingType = ::atol(strMeetingType);
					// MT_ONLINE=1 MT_ONLINE_WITH_OFFLINE=2 MT_ONLINE_FROM_OFFLINE=3 define in products\jabber-win\src\plugins\JabberMeetingPlugin\MeetingTab\MeetingData.h
					if (lMeetingType != 1 && lMeetingType != 2 && lMeetingType != 3)   return false;
				}
				else   return false;

				continue;
			}

			// meeting start time
			if (xmlStrEqual(pCurrent->name, (const xmlChar*)"StartTime"))
			{
				const char* strStartTime = (const char*)pCurrent->children->content;
				if (strStartTime != NULL)   lStartTime = ::atol(strStartTime);
				continue;
			}

			// meeting end time
			if (xmlStrEqual(pCurrent->name, (const xmlChar*)"EndTime"))
			{
				const char* strEndTime = (const char*)pCurrent->children->content;
				if (strEndTime != NULL)  lEndTime = ::atol(strEndTime);
				continue;
			}

			if (lMeetingType != -1 && lEndTime != 0 && lStartTime != 0)   return true;
		}

		return false;
	}

	bool JabberWerxCommonHelper::isInOnlineMeeting()
	{
		csf::ScopedReadRWLock lock(mLock);
		long long tmCur = JCFCoreUtils::TimeUtils::LocalNowInMicroseconds()/1000000;
		//CSFLogInfoS(JabberWerxCommonHelperLogger, "Call isOnline meeting " << tmCur);
		for (size_t i = 0; i < m_WebExMeetingTime.size(); i++)
		{
			//CSFLogInfoS(JabberWerxCommonHelperLogger, "TEST JabberWerxCommonHelper::isInOnlineMeeting." << tmCur << " "  << m_WebExMeetingTime[i].first << " " << m_WebExMeetingTime[i].second);
			if (tmCur >= m_WebExMeetingTime[i].first && tmCur <= m_WebExMeetingTime[i].second)
			{
				return true;
			}
		}
		return false;
	}

	void JabberWerxCommonHelper::ClearMeetingList()
	{
		csf::ScopedWriteRWLock lock(mLock);
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::ClearMeetingList Clear meeting time.");
		m_WebExMeetingTime.clear();
	}

	void JabberWerxCommonHelper::AddSharingInWebExMeetingStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddSharingInWebExMeetingStatus.");

		m_pJabberServiceProvider->addSharingInWebExMeetingStatus();
	}

	void JabberWerxCommonHelper::RemoveSharingInWebExMeetingStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::RemoveSharingInWebExMeetingStatus.");
		
		m_pJabberServiceProvider->removeSharingInWebExMeetingStatus();
	}

	void JabberWerxCommonHelper::AddInDSStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddInDSStatus.");
	
		m_pJabberServiceProvider->addInDSStatus();
	}

	void JabberWerxCommonHelper::RemoveInDSStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::RemoveInDSStatus.");
	
		m_pJabberServiceProvider->removeInDSStatus();
	}

	void JabberWerxCommonHelper::AddSharingInDSStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::AddSharingInDSStatus.");
	
		m_pJabberServiceProvider->addSharingInDSStatus();
	}

	void JabberWerxCommonHelper::RemoveSharingInDSStatus()
	{
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::RemoveSharingInDSStatus.");
		
		m_pJabberServiceProvider->removeSharingInDSStatus();
	}

	BOOL JabberWerxCommonHelper::IsCupMode()
	{
		BOOL bRet = m_pJabberServiceProvider->isCupMode();
		CSFLogDebugS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsCupMode : " << (bRet ? "true" : "false"));
		return bRet;
	}

	BOOL JabberWerxCommonHelper::IsPhoneMode()
	{
		BOOL bRet = m_pJabberServiceProvider->isPhoneMode();
		CSFLogDebugS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::IsPhoneMode : " << (bRet ? "true" : "false"));
		return bRet;
	}

	void JabberWerxCommonHelper::InitHelper( SMART_PTR_NS::shared_ptr<UnifiedFactory> UnifiedFactory )
	{
		m_UnifiedFactory = UnifiedFactory;

		//listen to the AdvancedPresence capability 

		SMART_PTR_NS::shared_ptr<PresenceService> ptrPresenceService = CDependence::getInstance()->GetUnifiedFactory()->getService<PresenceService>();
		if (NULL != ptrPresenceService)
		{
			CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::InitHelper, get PresenceService, to listen to getAdvancedPresenceNotifier");
//			ptrPresenceService->getCapabilities()->getPresenceServiceCapabilitiesNotifiers()->getAdvancedPresenceNotifier()->connect(boost::bind(&JabberWerxCommonHelper::OnCapabilityChanged, this));
		}
	}

	bool JabberWerxCommonHelper::IsJIDChanged()
	{
		static std::string OldJID = "";
		std::string NewJID = GetUserName();
		if (OldJID != NewJID)
		{
			OldJID = NewJID;
			return true;
		}
		else
		{
			return false;
		}
	}

	void JabberWerxCommonHelper::OnCapabilityChanged()
	{
//		SMART_PTR_NS::shared_ptr<PresenceService> ptrPS = CDependence::getInstance()->GetUnifiedFactory()->getService<PresenceService>();

//		if (NULL != ptrPS && ptrPS->getCapabilities()->getAdvancedPresence()/* && InAMeeting()*/)
//		{
//			//CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::OnCapabilityChanged, getAdvancedPresence:TRUE");
			//UpdateRichPresenceStatus(InAMeeting);
//		}
	}

	unsigned int JabberWerxCommonHelper::GetCalendarAutoRefreshTime()
	{
		unsigned int ret = 0;

		if (NULL != m_UnifiedFactory)
		{
			SMART_PTR_NS::shared_ptr<ConfigService> services = m_UnifiedFactory->getService<ConfigService>();
			if (NULL != services)
			{
				SMART_PTR_NS::shared_ptr<ConfigValue> ptrConfigValue = services->getConfig(L"CalendarAutoRefreshTime");
				if (NULL != ptrConfigValue)
				{
					ret = JCFCoreUtils::toUnsignedInt(ptrConfigValue->getValue(), 0);
				}
			}
		}
		CSFLogInfoS(JabberWerxCommonHelperLogger, "JabberWerxCommonHelper::GetCalendarAutoRefreshTime: " << ret);
		return ret;
	}
}

#endif